<?php
/**
 * Functions which enhance the theme by hooking into WordPress and Core theme Functions.
 *
 * @package Marpha
 */

/*----------------------------------------------------------------------
# Exit if accessed directly
-------------------------------------------------------------------------*/
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/*----------------------------------------------------------------------
# Adds custom classes to the array of body classes.
-------------------------------------------------------------------------*/
if ( !function_exists( 'marpha_body_classes' ) ) {
    /**
     * @param array $classes Classes for the body element.
     * @return array
     */
    function marpha_body_classes( $classes ) {

        // Adds a class of hfeed to non-singular pages.
        if ( ! is_singular() ) {
            $classes[] = 'hfeed';
        }

        // class for sticky sidebar
        if( get_theme_mod( 'marpha_breadcrumb_sticky_sidebar', false ) ){
            $classes[] = 'sticky-sidebar';
        }

        return apply_filters( 'marpha_body_classes', $classes );
    }
}
add_filter( 'body_class', 'marpha_body_classes' );

/*----------------------------------------------------------------------
# Add a pingback url auto-discovery header for single posts, pages, or attachments.
-------------------------------------------------------------------------*/
if ( !function_exists( 'marpha_pingback_header' ) ) {

    function marpha_pingback_header() {
        if ( is_singular() && pings_open() ) {
            echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
        }
    }
}
add_action( 'wp_head', 'marpha_pingback_header' );

/*----------------------------------------------------------------------
# Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
-------------------------------------------------------------------------*/
if ( !function_exists( 'marpha_page_menu_args' ) ) {
    function marpha_page_menu_args( $args ) {
        $args['show_home'] = true;
        return $args;
    };
}
add_filter( 'wp_page_menu_args', 'marpha_page_menu_args', 10, 1 );

/*----------------------------------------------------------------------
#  Returns correct ID
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_get_the_ID' ) ) {
    function marpha_get_the_ID() {

        // Default value is empty
        $id = get_the_ID();

        // Blog Page
        if ( is_home() && $page_for_posts = get_option( 'page_for_posts' ) ) {
            $id = $page_for_posts;
        }

        // Apply filters and return
        return apply_filters( 'marpha_post_id', absint( $id ) );

    }
}

/*----------------------------------------------------------------------
# Get custom permalink
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_get_permalink' ) ) :

    function marpha_get_permalink( $post_id = '' ) {
        // Apply filters and return
        return apply_filters( 'marpha_get_permalink', esc_url( get_permalink() ) );

    }
endif;

/*----------------------------------------------------------------------
# Prints HTML with meta information for the categories.
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_cat_links' ) ) {

    function marpha_cat_links( $before='', $after='', $post_id = false ) {

        if ( 'post' === get_post_type() ) {

            $cat_sep            = '<span class="item-sep vertical-line"></span>';

            $categories_list    = get_the_category_list( $cat_sep, '', $post_id ? $post_id : get_the_ID() );
            $output             = '';

            if ( $categories_list ) {

                $output .= '<div class="cat-links post-meta-item d-flex flex-wrap align-items-center">';

                $output .= $categories_list;
                $output .= '</div><!-- .cat-links -->';
            }

            // Filter
            $output = apply_filters( 'marpha_cat_links', $output );

            if ( ! empty( $output ) ) {
                echo $before . $output . $after;
            }

        }

    }

}

/*----------------------------------------------------------------------
# Prints HTML with meta information for the tags.
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_tags_links' ) ) {

    function marpha_tags_links( $before='', $after='' ) {
        if ( 'post' === get_post_type() ) {

            $tag_sep    = '<span class="item-sep comma"></span>';

            $tags_list  = get_the_tag_list( '', $tag_sep );
            $output     = '';

            if ( $tags_list ) {

                $output .= '<div class="tags-links post-meta-item d-flex flex-wrap align-items-center">';

                $output .= $tags_list;
                $output .= '</div><!-- .tags-links -->';
            }
            // Filter
            $output = apply_filters( 'marpha_tags_links', $output );

            if ( ! empty( $output ) ) {
                echo $before . $output . $after;
            }

        }

    }

}

/*----------------------------------------------------------------------
# Prints HTML with meta information for the author.
-------------------------------------------------------------------------*/
if ( !function_exists( 'marpha_post_author' ) ) {

    function marpha_post_author( $before = '', $after = '' ) {

        $output = '';
        $output .= '<div class="post-author post-meta-item d-flex flex-wrap align-items-center">';

        $output .= esc_html__( 'By', 'marpha' ) . ' <a href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . get_the_author() . '</a>';
        $output .= '</div>';

        // Filter
        $output = apply_filters( 'marpha_post_author', $output );

        if ( ! empty( $output ) ) {
            echo $before . $output . $after;
        }
    }
}

/*----------------------------------------------------------------------
# Prints HTML with meta information for the date.
-------------------------------------------------------------------------*/
if ( !function_exists( 'marpha_posted_date' ) ) {

    function marpha_posted_date( $before='', $after='' ) {

        $icon       = get_theme_mod( 'marpha_global_meta_date_icon', 'default' );

        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
        if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
            $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
        }

        $time_string = sprintf( $time_string,
            esc_attr( get_the_date( 'c' ) ),
            esc_html( get_the_date() ),
            esc_attr( get_the_modified_date( 'c' ) ),
            esc_html( get_the_modified_date() )
        );

        $posted_on = '<a href="' . esc_url( get_month_link(get_the_time('Y'), get_the_time('m')) ) . '" rel="bookmark">' . $time_string . '</a>';
        $output = '';
        $output .= '<div class="posted-on post-meta-item d-flex flex-wrap align-items-center">';

        $output .= $posted_on;
        $output .= '</div>';

        // Filter
        $output = apply_filters( 'marpha_posted_date', $output );

        if ( ! empty( $output ) ) {
            echo $before . $output . $after;
        }
    }
}

/*----------------------------------------------------------------------
# Add Action for the copyright information.
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_footer_copyright_information' ) ) {
    /**
     * Function to show the copyright information
     */
    function marpha_footer_copyright_information() {
        $default = esc_html__( 'Copyright All Right Reserved 2020', 'marpha' );
        printf( '<div class="site-info">%1$s | </span>%2$s <a href="%3$s" rel="designer" target="_blank">Precise Themes</a></div><!-- .site-info -->',
            esc_html( get_theme_mod( 'marpha_copyright_text', $default ) ),
            esc_html__( 'Designed by','marpha' ),
            esc_url( 'http://precisethemes.com/' )
        );
    }
}
add_action( 'marpha_footer_copyright', 'marpha_footer_copyright_information', 5 );

/*----------------------------------------------------------------------
# Returns sidebar layout
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_get_sidebar_layout' ) ) {
    function marpha_get_sidebar_layout( $post_id = '' ) {

        // Global Sidebar Layout
        $global_layout = get_theme_mod( 'marpha_sidebar_layout', 'right-sidebar' );

        // Get post ID
        $post_id = $post_id ? $post_id : marpha_get_the_ID();

        // Bail if blog page or archive page
        if( is_home() || is_archive() ) {

            // For all Archive Page
            $archive_layout = get_theme_mod( 'marpha_archive_sidebar_layout', 'right-sidebar' );

            // Check meta first to override and return (prevents filters from overriding meta)
            if ( $post_id && $meta = get_post_meta( $post_id, 'marpha_sidebar_layout', true ) ) {

                return apply_filters( 'marpha_get_sidebar_layout', esc_attr( $meta ) );

            }
            elseif ( $archive_layout !== 'default' ) {
                $global_layout = $archive_layout;

            }

        }

        // Bail if single page
        elseif ( is_404() ) {

            $global_layout = 'full-width';
        }

        // Bail if single page
        elseif ( is_page() || is_single() ) {


            // Check meta first to override and return (prevents filters from overriding meta)
            if ( $post_id && $meta = get_post_meta( $post_id, 'marpha_sidebar_layout', true ) ) {

                return apply_filters( 'marpha_get_sidebar_layout', esc_attr( $meta ) );

            }
        }

        // Apply filters and return
        return apply_filters( 'marpha_get_sidebar_layout', esc_attr( $global_layout ) );

    }
}

/*----------------------------------------------------------------------
# Primary Class
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_has_primary_content_class' ) ) {
    function marpha_has_primary_content_class() {

        $sidebar_layout = marpha_get_sidebar_layout();

        if ( $sidebar_layout == 'right-sidebar' ) {
            $primary_class = 'order-1';
        }
        elseif ( $sidebar_layout == 'left-sidebar' ) {
            $primary_class = 'order-2';
        }
        else {
            $primary_class = 'full-width';
        }
        // Apply filters and return
        return apply_filters( 'marpha_has_primary_content_class', esc_attr( $primary_class ));

    }
}

/*----------------------------------------------------------------------
# Secondary Class
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_has_secondary_content_class' ) ) {
    function marpha_has_secondary_content_class() {

        $sidebar_layout = marpha_get_sidebar_layout();

        if ( $sidebar_layout == 'right-sidebar' ) {
            $secondary_class = 'order-2';
        } elseif ( $sidebar_layout == 'left-sidebar' ) {
            $secondary_class = 'order-1';
        } else {
            $secondary_class = $sidebar_layout;
        }
        // Apply filters and return
        return apply_filters( 'marpha_has_secondary_content_class', esc_attr( $secondary_class ) );
    }
}

/*----------------------------------------------------------------------
# Sidebar layout options
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_content_layouts' ) ) :
    /**
     * Returns array of content layouts for page or page.
     */
    function marpha_content_layouts( $output = array() ) {

        $output['left-sidebar']     = esc_html__( 'Left Sidebar', 'marpha' );
        $output['full-width']       = esc_html__( 'Full Width', 'marpha' );
        $output['right-sidebar']    = esc_html__( 'Right Sidebar', 'marpha' );
        return $output;
    }
endif;

/*----------------------------------------------------------------------
# Social Profiles lists
-------------------------------------------------------------------------*/
if ( ! function_exists( 'marpha_social_profiles' ) ) :
    /**
     * Returns array of social profiles.
     */
    function marpha_social_profiles( $output = array() ) {
        $output['fa-behance']       = esc_html__( 'Behance', 'marpha' );
        $output['fa-delicious']     = esc_html__( 'Delicious', 'marpha' );
        $output['fa-digg']          = esc_html__( 'Digg', 'marpha' );
        $output['fa-dribbble']      = esc_html__( 'Dribbble', 'marpha' );
        $output['fa-facebook']      = esc_html__( 'Facebook', 'marpha' );
        $output['fa-flickr']        = esc_html__( 'Flickr', 'marpha' );
        $output['fa-foursquare']    = esc_html__( 'Foursquare', 'marpha' );
        $output['fa-github']        = esc_html__( 'Github', 'marpha' );
        $output['fa-google-plus']   = esc_html__( 'Google Plus', 'marpha' );
        $output['fa-instagram']     = esc_html__( 'Instagram', 'marpha' );
        $output['fa-linkedin']      = esc_html__( 'LinkedIn', 'marpha' );
        $output['fa-envelope']      = esc_html__( 'Mail', 'marpha' );
        $output['fa-medium']        = esc_html__( 'Medium', 'marpha' );
        $output['fa-pinterest']     = esc_html__( 'Pinterest', 'marpha' );
        $output['fa-reddit']        = esc_html__( 'Reddit', 'marpha' );
        $output['fa-skype']         = esc_html__( 'Skype', 'marpha' );
        $output['fa-slack']         = esc_html__( 'Slack', 'marpha' );
        $output['fa-stackoverflow'] = esc_html__( 'Stackoverflow', 'marpha' );
        $output['fa-twitter']       = esc_html__( 'Twitter', 'marpha' );
        $output['fa-tumblr']        = esc_html__( 'Tumblr', 'marpha' );
        $output['fa-vimeo']         = esc_html__( 'Vimeo', 'marpha' );
        $output['fa-youtube']       = esc_html__( 'YouTube', 'marpha' );
        return $output;
    }
endif;


/*--------------------------------------------------------------
# Returns posts latest post / category
--------------------------------------------------------------*/
if( !function_exists( 'marpha_get_featured_news_post' ) ):
    function marpha_get_featured_news_post( $type, $cat_id, $post_to_display = -1 ){
        $posts = array();
        if( 'latest' == $type ){
            $recent_posts = wp_get_recent_posts(array(
                'numberposts' => $post_to_display,
                'post_status' => 'publish'
            ));

            foreach ( $recent_posts as $post) {
                $posts[] = $post[ 'ID' ]; 
            }
        }elseif( 'category' == $type ){     
            $args = array(
                'post_type' => 'post',
                'posts_per_page' => $post_to_display,
                'orderby' => 'post__in',
                'ignore_sticky_posts' => 1,
            );
            if( $cat_id ){
                $args[ 'cat' ] = $cat_id; 
            }

            $query = new WP_Query( $args );
            if ( $query->have_posts() ) {
                while ( $query->have_posts() ) {
                    $query->the_post();
                    $posts[] = get_the_ID();
                }
                wp_reset_postdata();
            }
        }
        if( empty( $posts ) ){
            return false;
        }else{
            return $posts;
        }
    }
endif;

/*--------------------------------------------------------------
# Returns breadcrumb
--------------------------------------------------------------*/
add_action( 'marpha_after_header', 'marpha_get_breadcrumb' );
if( !function_exists( 'marpha_get_breadcrumb' ) ):
    function marpha_get_breadcrumb(){
        if( 
            # don't load it in 404 page
            is_404() ||
            # don't load it if home page
            is_home() ||
            # dont load statixc front page
            is_front_page() ||
            # load if disabled by customizer
            !get_theme_mod( 'marpha_breadcrumb_on_off', true )
        ){ 
            return;
        }

        $breadcrumb_args = array(
            'container'   => 'div',
            'show_browse' => false,
        );
        marpha_breadcrumb_trail( $breadcrumb_args );
    }
endif;